<?php
/**
 * Template Name: Avulsos — Cadastro & Lista (sem AJAX, paginação 50)
 */
if (!defined('ABSPATH')) exit;
date_default_timezone_set('America/Manaus');

get_header();

/* =========================
 * CONFIG / HELPERS
 * ========================= */
$NONCE_ACTION = 'avulsos_sem_ajax';
$PER_PAGE     = 20; // <<<<<< PAGINAÇÃO: 50 por página

define('AVULSOS_ACF_KEYS', [
  'tipo'      => 'field_67250de673988',
  'dinheiro'  => 'field_672511bb9f3f3',
  'pix'       => 'field_672512059f3f4',
  'debito'    => 'field_672512119f3f5',
  'credito'   => 'field_672512229f3f6',
  'historico' => 'field_67250e46a8106',
]);
$FIELD_NAMES_FALLBACK = [
  'tipo'      => 'tipo',
  'dinheiro'  => 'avulso_dinheiro',
  'pix'       => 'avulso_pix',
  'debito'    => 'avulso_debito',
  'credito'   => 'avulso_credito',
  'historico' => 'historico',
];
$ALLOWED_TIPOS = ['Saida','Entrada','Cancelado'];

function avu_br2f($v){
  if ($v === '' || $v === null) return 0.0;
  if (is_numeric($v)) return (float)$v;
  $v = str_replace(['.', ' '], '', (string)$v);
  $v = str_replace(',', '.', $v);
  return is_numeric($v) ? (float)$v : 0.0;
}
function avu_f2br($f){ return number_format((float)$f, 2, ',', '.'); }
function avu_update_field($key, $value, $post_id, $fallback_name){
  if (function_exists('update_field')) return update_field($key, $value, $post_id);
  return update_post_meta($post_id, $fallback_name, $value);
}
function avu_get_field($key, $post_id, $fallback_name){
  if (function_exists('get_field')) return get_field($fallback_name, $post_id);
  return get_post_meta($post_id, $fallback_name, true);
}

if (!is_user_logged_in() || !current_user_can('edit_posts')) {
  wp_die('Você precisa estar logado com permissão para editar.');
}

$base_url = get_permalink();
$notice   = isset($_GET['notice']) ? sanitize_text_field($_GET['notice']) : '';
$error    = '';

/* Página atual (server-side) */
$paged = get_query_var('paged');
if (!$paged) {
  $p = isset($_GET['pg']) ? (int)$_GET['pg'] : 1;
  $paged = max(1, $p);
}

/* =========================
 * POST: CREATE / UPDATE / DELETE (sem AJAX)
 * ========================= */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['avu_action'])) {
  if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], $NONCE_ACTION)) {
    $error = 'Falha de segurança. Atualize a página e tente novamente.';
  } else {
    $acao = sanitize_text_field($_POST['avu_action']);

    // Captura campos comuns
    $tipo      = sanitize_text_field($_POST['tipo'] ?? '');
    $historico = wp_kses_post($_POST['historico'] ?? '');

    $dinheiro = sanitize_text_field($_POST['dinheiro'] ?? '0,00');
    $pix      = sanitize_text_field($_POST['pix'] ?? '0,00');
    $debito   = sanitize_text_field($_POST['debito'] ?? '0,00');
    $credito  = sanitize_text_field($_POST['credito'] ?? '0,00');

    if (!in_array($tipo, $ALLOWED_TIPOS, true)) $tipo = 'Entrada';
    $pg_back = isset($_POST['pg']) ? max(1, (int)$_POST['pg']) : 1;

    if ($acao === 'create') {
      if ($tipo === '' || trim(strip_tags($historico)) === '') {
        $error = 'Informe Tipo e Histórico.';
      } else {
        $title = sprintf('Avulso %s — %s', $tipo, wp_trim_words(wp_strip_all_tags($historico), 6, '…'));
        $pid = wp_insert_post([
          'post_type'   => 'avulsos',
          'post_status' => 'publish',
          'post_title'  => $title,
          'post_content'=> '',
        ], true);

        if (is_wp_error($pid)) {
          $error = 'Erro ao criar: ' . $pid->get_error_message();
        } else {
          avu_update_field(AVULSOS_ACF_KEYS['tipo'],      $tipo,     $pid, 'tipo');
          avu_update_field(AVULSOS_ACF_KEYS['dinheiro'],  $dinheiro, $pid, 'avulso_dinheiro');
          avu_update_field(AVULSOS_ACF_KEYS['pix'],       $pix,      $pid, 'avulso_pix');
          avu_update_field(AVULSOS_ACF_KEYS['debito'],    $debito,   $pid, 'avulso_debito');
          avu_update_field(AVULSOS_ACF_KEYS['credito'],   $credito,  $pid, 'avulso_credito');
          avu_update_field(AVULSOS_ACF_KEYS['historico'], $historico,$pid, 'historico');

          // Após criar, vai para a página 1 (onde o novo costuma aparecer)
          wp_safe_redirect(add_query_arg(['notice'=>'created','pg'=>1], $base_url));
          exit;
        }
      }
    }

    if ($acao === 'update') {
      $post_id = isset($_POST['post_id']) ? (int)$_POST['post_id'] : 0;
      if ($post_id <= 0 || get_post_type($post_id) !== 'avulsos') {
        $error = 'Registro inválido.';
      } else {
        $title = sprintf('Avulso %s — %s', $tipo, wp_trim_words(wp_strip_all_tags($historico), 6, '…'));
        wp_update_post([
          'ID'         => $post_id,
          'post_title' => $title,
        ]);
        avu_update_field(AVULSOS_ACF_KEYS['tipo'],      $tipo,     $post_id, 'tipo');
        avu_update_field(AVULSOS_ACF_KEYS['dinheiro'],  $dinheiro, $post_id, 'avulso_dinheiro');
        avu_update_field(AVULSOS_ACF_KEYS['pix'],       $pix,      $post_id, 'avulso_pix');
        avu_update_field(AVULSOS_ACF_KEYS['debito'],    $debito,   $post_id, 'avulso_debito');
        avu_update_field(AVULSOS_ACF_KEYS['credito'],   $credito,  $post_id, 'avulso_credito');
        avu_update_field(AVULSOS_ACF_KEYS['historico'], $historico,$post_id, 'historico');

        wp_safe_redirect(add_query_arg(['notice'=>'updated','pg'=>$pg_back], $base_url));
        exit;
      }
    }

    if ($acao === 'delete') {
      $post_id = isset($_POST['post_id']) ? (int)$_POST['post_id'] : 0;
      if ($post_id <= 0 || get_post_type($post_id) !== 'avulsos') {
        $error = 'Registro inválido.';
      } else {
        wp_trash_post($post_id);
        wp_safe_redirect(add_query_arg(['notice'=>'deleted','pg'=>$pg_back], $base_url));
        exit;
      }
    }
  }
}

/* =========================
 * QUERY: LISTA (paginação servidor)
 * ========================= */
$q = new WP_Query([
  'post_type'      => 'avulsos',
  'posts_per_page' => $PER_PAGE,
  'post_status'    => 'publish',
  'orderby'        => 'date',
  'order'          => 'DESC',
  'fields'         => 'ids',
  'paged'          => $paged,
  'no_found_rows'  => false, // NECESSÁRIO PARA $q->max_num_pages
]);
$total_pages = max(1, (int)$q->max_num_pages);
$found_posts = (int)$q->found_posts;
$from_i      = ($paged - 1) * $PER_PAGE + 1;
$to_i        = min($found_posts, $paged * $PER_PAGE);

// Útil para links de paginação
$base_no_pg = remove_query_arg('pg', $base_url);
$mkurl = function($n) use ($base_no_pg) {
  return esc_url(add_query_arg('pg', max(1,(int)$n), $base_no_pg));
};

?>
<!-- ====== CSS/JS (Bootstrap 5 + DataTables) ====== -->
<link rel="stylesheet">
<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
<link href="https://cdn.datatables.net/1.13.8/css/dataTables.bootstrap5.min.css" rel="stylesheet">

<div class="container my-4">
  <div class="d-flex align-items-center justify-content-between mb-3">
    <h3 class="mb-0">Avulsos — Cadastro</h3>
    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#modalNovo">
      <i class="bi bi-plus-circle"></i> Novo Avulso
    </button>
  </div>

  <?php if ($notice === 'created'): ?>
    <div class="alert alert-success">Registro criado com sucesso.</div>
  <?php elseif ($notice === 'updated'): ?>
    <div class="alert alert-success">Registro atualizado com sucesso.</div>
  <?php elseif ($notice === 'deleted'): ?>
    <div class="alert alert-warning">Registro enviado para a lixeira.</div>
  <?php endif; ?>

  <?php if ($error): ?>
    <div class="alert alert-danger"><?= esc_html($error); ?></div>
  <?php endif; ?>

  <div class="d-flex justify-content-between align-items-center mb-2">
    <div>
      <span class="badge text-bg-info">Página <?= (int)$paged; ?> de <?= (int)$total_pages; ?></span>
      <span class="ms-2 text-muted">Exibindo <?= (int)$from_i; ?>–<?= (int)$to_i; ?> de <?= (int)$found_posts; ?></span>
    </div>
  </div>

  <div class="table-responsive">
    <table id="tAvulsos" class="table table-striped table-bordered align-middle">
      <thead class="table-light">
        <tr>
          <th>ID</th>
          <th>Data</th>
          <th>Tipo</th>
          <th>Histórico</th>
          <th>Dinheiro</th>
          <th>Pix</th>
          <th>Débito</th>
          <th>Crédito</th>
          <th>Total</th>
          <th style="min-width:130px;">Ações</th>
        </tr>
      </thead>
      <tbody>
        <?php foreach ($q->posts as $pid):
          $tipo      = (string) avu_get_field(AVULSOS_ACF_KEYS['tipo'],      $pid, 'tipo');
          $dinheiro  = (string) avu_get_field(AVULSOS_ACF_KEYS['dinheiro'],  $pid, 'avulso_dinheiro');
          $pix       = (string) avu_get_field(AVULSOS_ACF_KEYS['pix'],       $pid, 'avulso_pix');
          $debito    = (string) avu_get_field(AVULSOS_ACF_KEYS['debito'],    $pid, 'avulso_debito');
          $credito   = (string) avu_get_field(AVULSOS_ACF_KEYS['credito'],   $pid, 'avulso_credito');
          $historico = (string) avu_get_field(AVULSOS_ACF_KEYS['historico'], $pid, 'historico');

          $dinheiro = $dinheiro !== '' ? $dinheiro : '0,00';
          $pix      = $pix      !== '' ? $pix      : '0,00';
          $debito   = $debito   !== '' ? $debito   : '0,00';
          $credito  = $credito  !== '' ? $credito  : '0,00';
          $total    = avu_f2br( avu_br2f($dinheiro) + avu_br2f($pix) + avu_br2f($debito) + avu_br2f($credito) );

          $modalId = 'modalEdit_'.$pid;
        ?>
        <tr>
          <td><?= (int)$pid; ?></td>
          <td><?= esc_html(get_the_date('d/m/Y H:i', $pid)); ?></td>
          <td>
            <?php if ($tipo==='Saida'): ?>
              <span class="badge text-bg-danger">Saída</span>
            <?php elseif ($tipo==='Entrada'): ?>
              <span class="badge text-bg-success">Entrada</span>
            <?php else: ?>
              <span class="badge text-bg-secondary">Cancelado</span>
            <?php endif; ?>
          </td>
          <td><?= wp_kses_post(nl2br($historico ?: '-')); ?></td>
          <td><?= esc_html($dinheiro); ?></td>
          <td><?= esc_html($pix); ?></td>
          <td><?= esc_html($debito); ?></td>
          <td><?= esc_html($credito); ?></td>
          <td><strong><?= esc_html($total); ?></strong></td>
          <td class="text-nowrap">
            <button class="btn btn-sm btn-outline-primary" data-bs-toggle="modal" data-bs-target="#<?= esc_attr($modalId); ?>">
              Editar
            </button>
            <form method="post" class="d-inline" onsubmit="return confirm('Enviar para lixeira este registro?');">
              <?php wp_nonce_field($NONCE_ACTION); ?>
              <input type="hidden" name="avu_action" value="delete">
              <input type="hidden" name="post_id" value="<?= (int)$pid; ?>">
              <input type="hidden" name="pg" value="<?= (int)$paged; ?>">
              <button class="btn btn-sm btn-outline-danger" type="submit">Excluir</button>
            </form>
          </td>
        </tr>

        <!-- ===== Modal Editar (sem AJAX) ===== -->
        <div class="modal fade" id="<?= esc_attr($modalId); ?>" tabindex="-1" aria-hidden="true">
          <div class="modal-dialog modal-lg modal-dialog-scrollable">
            <div class="modal-content">
              <form method="post">
                <div class="modal-header">
                  <h5 class="modal-title">Editar Avulso #<?= (int)$pid; ?></h5>
                  <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
                </div>
                <div class="modal-body">
                  <?php wp_nonce_field($NONCE_ACTION); ?>
                  <input type="hidden" name="avu_action" value="update">
                  <input type="hidden" name="post_id" value="<?= (int)$pid; ?>">
                  <input type="hidden" name="pg" value="<?= (int)$paged; ?>">

                  <div class="row g-3">
                    <div class="col-md-4">
                      <label class="form-label">Tipo</label>
                      <select name="tipo" class="form-select" required>
                        <?php foreach ($ALLOWED_TIPOS as $opt): ?>
                          <option value="<?= esc_attr($opt); ?>" <?= selected($tipo, $opt, false); ?>><?= esc_html($opt); ?></option>
                        <?php endforeach; ?>
                      </select>
                    </div>

                    <div class="col-md-2">
                      <label class="form-label">Dinheiro</label>
                      <input type="text" name="dinheiro" class="form-control js-money" value="<?= esc_attr($dinheiro); ?>">
                    </div>
                    <div class="col-md-2">
                      <label class="form-label">Pix</label>
                      <input type="text" name="pix" class="form-control js-money" value="<?= esc_attr($pix); ?>">
                    </div>
                    <div class="col-md-2">
                      <label class="form-label">Débito</label>
                      <input type="text" name="debito" class="form-control js-money" value="<?= esc_attr($debito); ?>">
                    </div>
                    <div class="col-md-2">
                      <label class="form-label">Crédito</label>
                      <input type="text" name="credito" class="form-control js-money" value="<?= esc_attr($credito); ?>">
                    </div>

                    <div class="col-12">
                      <label class="form-label">Histórico</label>
                      <textarea name="historico" rows="4" class="form-control" required><?= esc_textarea($historico); ?></textarea>
                    </div>
                  </div>
                </div>
                <div class="modal-footer">
                  <button class="btn btn-secondary" type="button" data-bs-dismiss="modal">Cancelar</button>
                  <button class="btn btn-primary" type="submit">Salvar alterações</button>
                </div>
              </form>
            </div>
          </div>
        </div>
        <?php endforeach; ?>
      </tbody>
      <tfoot>
        <tr>
          <th colspan="9" class="text-end">—</th>
          <th></th>
        </tr>
      </tfoot>
    </table>
  </div>

  <!-- ====== Paginação (servidor, 50 por página) ====== -->
  <?php if ($total_pages > 1): ?>
    <nav aria-label="Paginação Avulsos">
      <ul class="pagination justify-content-center mt-3">
        <li class="page-item <?= $paged<=1?'disabled':''; ?>">
          <a class="page-link" href="<?= $mkurl(1); ?>">« Primeira</a>
        </li>
        <li class="page-item <?= $paged<=1?'disabled':''; ?>">
          <a class="page-link" href="<?= $mkurl($paged-1); ?>">‹ Anterior</a>
        </li>

        <?php
          $start = max(1, $paged-2);
          $end   = min($total_pages, $paged+2);
          if ($start > 1) {
            echo '<li class="page-item"><a class="page-link" href="'.$mkurl(1).'">1</a></li>';
            if ($start > 2) echo '<li class="page-item disabled"><span class="page-link">…</span></li>';
          }
          for ($i=$start; $i<=$end; $i++) {
            $active = $i===$paged ? 'active' : '';
            echo '<li class="page-item '.$active.'"><a class="page-link" href="'.$mkurl($i).'">'.$i.'</a></li>';
          }
          if ($end < $total_pages) {
            if ($end < $total_pages-1) echo '<li class="page-item disabled"><span class="page-link">…</span></li>';
            echo '<li class="page-item"><a class="page-link" href="'.$mkurl($total_pages).'">'.$total_pages.'</a></li>';
          }
        ?>

        <li class="page-item <?= $paged>=$total_pages?'disabled':''; ?>">
          <a class="page-link" href="<?= $mkurl($paged+1); ?>">Próxima ›</a>
        </li>
        <li class="page-item <?= $paged>=$total_pages?'disabled':''; ?>">
          <a class="page-link" href="<?= $mkurl($total_pages); ?>">Última »</a>
        </li>
      </ul>
    </nav>
  <?php endif; ?>
</div>

<!-- ===== Modal Novo (sem AJAX) ===== -->
<div class="modal fade" id="modalNovo" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-scrollable">
    <div class="modal-content">
      <form method="post">
        <div class="modal-header">
          <h5 class="modal-title">Novo Avulso</h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
        </div>
        <div class="modal-body">
          <?php wp_nonce_field($NONCE_ACTION); ?>
          <input type="hidden" name="avu_action" value="create">
          <input type="hidden" name="pg" value="<?= (int)$paged; ?>">

          <div class="row g-3">
            <div class="col-md-4">
              <label class="form-label">Tipo</label>
              <select name="tipo" class="form-select" required>
                <?php foreach ($ALLOWED_TIPOS as $opt): ?>
                  <option value="<?= esc_attr($opt); ?>"><?= esc_html($opt); ?></option>
                <?php endforeach; ?>
              </select>
            </div>

            <div class="col-md-2">
              <label class="form-label">Dinheiro</label>
              <input type="text" name="dinheiro" class="form-control js-money" value="0,00">
            </div>
            <div class="col-md-2">
              <label class="form-label">Pix</label>
              <input type="text" name="pix" class="form-control js-money" value="0,00">
            </div>
            <div class="col-md-2">
              <label class="form-label">Débito</label>
              <input type="text" name="debito" class="form-control js-money" value="0,00">
            </div>
            <div class="col-md-2">
              <label class="form-label">Crédito</label>
              <input type="text" name="credito" class="form-control js-money" value="0,00">
            </div>

            <div class="col-12">
              <label class="form-label">Histórico</label>
              <textarea name="historico" rows="4" class="form-control" required></textarea>
            </div>
          </div>
        </div>
        <div class="modal-footer">
          <button class="btn btn-secondary" type="button" data-bs-dismiss="modal">Cancelar</button>
          <button class="btn btn-primary" type="submit">Salvar</button>
        </div>
      </form>
    </div>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.datatables.net/1.13.8/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.8/js/dataTables.bootstrap5.min.js"></script>

<script>
  // Máscara simples de moeda pt-BR
  (function(){
    function formatBR(el){
      let v = el.value.replace(/[^\d]/g, '');
      if (!v) { el.value = '0,00'; return; }
      while (v.length < 3) v = '0' + v;
      let intPart = v.slice(0, -2);
      let decPart = v.slice(-2);
      intPart = intPart.replace(/\B(?=(\d{3})+(?!\d))/g, '.');
      el.value = intPart + ',' + decPart;
    }
    document.querySelectorAll('.js-money').forEach(function(inp){
      inp.addEventListener('input', function(){ formatBR(inp); });
      inp.addEventListener('blur',  function(){ formatBR(inp); });
    });
  })();

  // DataTables: sem paginação (quem pagina é o servidor em 50/pg)
  $(function(){
    $('#tAvulsos').DataTable({
      paging: false,
      lengthChange: false,
      info: false,
      order: [[1, 'desc']],
      language: { url: 'https://cdn.datatables.net/plug-ins/1.13.8/i18n/pt-BR.json' },
      footerCallback: function ( row, data, start, end, display ) {
        let api = this.api();
        let parseBR = function(s){
          if(!s) return 0;
          s = String(s).replace(/\./g,'').replace(',','.');
          let n = parseFloat(s);
          return isNaN(n)?0:n;
        };
        let total = api.column(8, {page:'current'}).data().reduce(function(sum, v){
          return sum + parseBR(String(v).replace(/<[^>]*>/g,''));
        }, 0);
        let fmt = (total).toLocaleString('pt-BR', {minimumFractionDigits:2, maximumFractionDigits:2});
        $(api.table().footer()).find('th:last').html('<strong>Soma pág: '+fmt+'</strong>');
      }
    });
  });
</script>

<?php get_footer(); ?>
